"use client"

import { Badge } from "@/components/ui/badge"
import { Button } from "@/components/ui/button"
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import { Terminal, Code, Server, Shield, Heart, Zap } from "lucide-react"
import Image from "next/image"
import Link from "next/link"
import { useEffect, useState } from "react"
import { LanguageSelector } from "@/components/LanguageSelector"
import { useTranslation } from "@/hooks/useTranslation"
import translations from "@/i18n/translations"

export default function Portfolio() {
  const { language, mounted } = useTranslation()
  const [glitchText, setGlitchText] = useState("Naïm.dev")
  const [matrixRain, setMatrixRain] = useState<string[]>([])

  const t = (key: keyof typeof translations.fr): string => {
    return translations[language]?.[key] || translations.fr[key] || key
  }

  useEffect(() => {
    if (mounted) {
      setGlitchText(t("heroTitle"))
    }
  }, [language, mounted])

  useEffect(() => {
    // Glitch effect for title
    const glitchInterval = setInterval(() => {
      const glitchChars = "!@#$%^&*()_+-=[]{}|;:,.<>?"
      const original = mounted ? t("heroTitle") : "Naïm.dev"
      let glitched = ""

      for (let i = 0; i < original.length; i++) {
        if (Math.random() < 0.1) {
          glitched += glitchChars[Math.floor(Math.random() * glitchChars.length)]
        } else {
          glitched += original[i]
        }
      }

      setGlitchText(glitched)
      setTimeout(() => setGlitchText(original), 100)
    }, 3000)

    // Matrix rain effect
    const matrixChars = "01アイウエオカキクケコサシスセソタチツテトナニヌネノハヒフヘホマミムメモヤユヨラリルレロワヲン"
    const rainInterval = setInterval(() => {
      const newRain = Array.from({ length: 20 }, () => matrixChars[Math.floor(Math.random() * matrixChars.length)])
      setMatrixRain(newRain)
    }, 200)

    return () => {
      clearInterval(glitchInterval)
      clearInterval(rainInterval)
    }
  }, [mounted])

  const isRTL = language === "ar"

  return (
    <div className={`min-h-screen bg-black relative overflow-hidden ${isRTL ? "rtl" : "ltr"}`}>
      <LanguageSelector />

      {/* Matrix Rain Background */}
      <div className="fixed inset-0 pointer-events-none z-0">
        <div className="absolute inset-0 bg-gradient-to-br from-cyan-900/20 via-purple-900/20 to-pink-900/20"></div>
        {matrixRain.map((char, i) => (
          <div
            key={i}
            className="absolute text-green-400/30 font-mono text-sm animate-pulse"
            style={{
              left: `${(i * 5) % 100}%`,
              top: `${Math.random() * 100}%`,
              animationDelay: `${i * 0.1}s`,
            }}
          >
            {char}
          </div>
        ))}
      </div>

      {/* Cyber Grid Overlay */}
      <div className="fixed inset-0 pointer-events-none z-0">
        <div className="absolute inset-0 bg-[linear-gradient(rgba(0,255,255,0.03)_1px,transparent_1px),linear-gradient(90deg,rgba(0,255,255,0.03)_1px,transparent_1px)] bg-[size:50px_50px]"></div>
      </div>

      {/* Hero Section */}
      <section className="relative min-h-screen flex items-center justify-center overflow-hidden z-10">
        <div className="absolute inset-0 bg-gradient-to-br from-black/80 via-cyan-900/40 to-purple-900/60 z-10"></div>
        <Image
          src="/images/hero-anime.png"
          alt="Hero background"
          fill
          className="object-cover object-center opacity-60"
          priority
          quality={100}
          sizes="100vw"
        />

        <div className="relative z-20 text-center text-white max-w-5xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="mb-6">
            <div className="relative mb-8">
              <Terminal className="w-16 h-16 mx-auto mb-4 text-cyan-400 animate-pulse drop-shadow-[0_0_20px_rgba(0,255,255,0.5)]" />
              <div className="absolute -inset-4 bg-cyan-400/20 rounded-full blur-xl animate-pulse"></div>
            </div>

            <h1 className="text-4xl sm:text-6xl md:text-7xl lg:text-8xl font-bold mb-6 relative">
              <span className="bg-gradient-to-r from-cyan-400 via-purple-400 to-pink-400 bg-clip-text text-transparent animate-pulse drop-shadow-[0_0_30px_rgba(0,255,255,0.5)]">
                {glitchText}
              </span>
              <div className="absolute -inset-2 bg-gradient-to-r from-cyan-400/20 to-purple-400/20 blur-2xl -z-10"></div>
            </h1>

            <div className="relative mb-8">
              <p className="text-lg sm:text-xl md:text-2xl mb-6 text-gray-300 max-w-3xl mx-auto font-mono">
                <span className="text-cyan-400">{t("heroStatus")}</span> {t("heroSubtitle")}
              </p>
              <div className="h-px bg-gradient-to-r from-transparent via-cyan-400 to-transparent animate-pulse"></div>
            </div>
          </div>

          <div className="bg-black/90 backdrop-blur-sm rounded-lg p-6 mb-8 border border-cyan-400/50 max-w-2xl mx-auto relative overflow-hidden">
            <div className="absolute inset-0 bg-gradient-to-r from-cyan-400/10 to-purple-400/10 animate-pulse"></div>
            <div className="relative z-10">
              <p className="text-base sm:text-lg mb-4 font-mono">
                <span className="text-red-400 animate-pulse">sudo rm -rf /windows</span>
                <span className="text-green-400 ml-2 animate-bounce">✓</span>
              </p>
              <p className="text-green-400 font-mono text-sm sm:text-base mb-2">
                <span className="text-cyan-400">&gt;</span> {t("heroTerminal1")}
              </p>
              <p className="text-blue-400 font-mono text-sm sm:text-base mb-2">
                <span className="text-cyan-400">&gt;</span> {t("heroTerminal2")}
              </p>
              <p className="text-purple-400 font-mono text-sm sm:text-base">
                <span className="text-cyan-400">&gt;</span> {t("heroTerminal3")}
              </p>
            </div>
            <div className="absolute top-0 left-0 w-full h-px bg-gradient-to-r from-cyan-400 to-purple-400 animate-pulse"></div>
          </div>

          <Button
            asChild
            size="lg"
            className="bg-gradient-to-r from-cyan-600 to-purple-600 hover:from-cyan-700 hover:to-purple-700 text-white border border-cyan-400/50 shadow-[0_0_20px_rgba(0,255,255,0.3)] hover:shadow-[0_0_30px_rgba(0,255,255,0.5)] transition-all duration-300 flex items-center justify-center mx-auto"
          >
            <Link href="/projects">
              <Code className="w-5 h-5 mr-2" />
              <span className="font-mono">ACCÉDER_AU_SYSTÈME</span>
            </Link>
          </Button>
        </div>
      </section>

      {/* About Section */}
      <section id="about" className="py-20 px-4 relative z-10">
        <div className="max-w-6xl mx-auto">
          <h2 className="text-4xl font-bold text-center mb-12 text-white font-mono">
            <span className="text-cyan-400">[</span>
            {t("aboutTitle")}
            <span className="text-cyan-400">]</span>
          </h2>

          <div className="grid md:grid-cols-2 gap-12 items-center">
            <div>
              <Card className="bg-black/80 border-2 border-cyan-400/50 backdrop-blur-sm relative overflow-hidden">
                <div className="absolute inset-0 bg-gradient-to-br from-cyan-400/10 to-purple-400/10"></div>
                <div className="absolute top-0 left-0 w-full h-px bg-gradient-to-r from-cyan-400 to-purple-400 animate-pulse"></div>
                <CardHeader>
                  <CardTitle className="text-cyan-400 flex items-center font-mono">
                    <Shield className="w-6 h-6 mr-2 animate-pulse" />
                    <span className="text-green-400">[</span>
                    {t("aboutCardTitle")}
                    <span className="text-green-400">]</span>
                  </CardTitle>
                </CardHeader>
                <CardContent className="text-gray-300 relative z-10">
                  <p className="mb-4 font-mono text-sm leading-relaxed">{t("aboutText")}</p>
                  <div className="bg-black/60 rounded p-3 border border-green-400/30 mb-4">
                    <p className="text-green-400 font-mono text-sm mb-2">
                      <span className="text-cyan-400">$</span> whoami
                      <br />
                      <span className="text-yellow-400">je s'appelle root</span>
                    </p>
                    <p className="text-cyan-400 font-mono text-sm">
                      <span className="text-cyan-400">$</span> uptime --since-consistent-coding 
                      <br />
                      <span className="text-green-400">3 years, 365 days/year, 24/7</span>
                    </p>
                  </div>
                  <p className="text-yellow-400 text-sm mt-3 font-mono border border-yellow-400/30 p-2 rounded">
                    <span className="text-cyan-400">💡 [INFO]</span> {t("aboutLearning")}
                  </p>
                  <p className="text-purple-400 text-sm mt-2 font-mono border border-purple-400/30 p-2 rounded">
                    <span className="text-cyan-400">🎯 [TARGET]</span> Objectif devenir le meilleur partout car ma passion me le demande
                  </p>
                </CardContent>
              </Card>
            </div>

            <div className="relative">
              <div className="absolute -inset-4 bg-gradient-to-r from-cyan-400/20 to-purple-400/20 rounded-lg blur-xl animate-pulse"></div>
              <Image
                src="/images/linux-waifu.jpeg"
                alt="Linux Waifu"
                width={400}
                height={400}
                className="rounded-lg shadow-2xl border-2 border-cyan-400/50 relative z-10"
              />

            </div>
          </div>
        </div>
      </section>

      {/* Journey Section */}
      <section id="experience" className="py-20 px-4 relative z-10">
        <div className="max-w-4xl mx-auto">
          <h2 className="text-4xl font-bold text-center mb-12 text-white font-mono">
            <span className="text-cyan-400">[</span>
            {t("journeyTitle")}
            <span className="text-cyan-400">]</span>
          </h2>

          <div className="space-y-8">
            {[
              {
                age: 12,
                year: 2018,
                title: "INIT_CODING",
                desc: t("journey2018"),
              },
              {
                age: 14,
                year: 2020,
                title: "PYTHON_FUN_MODE",
                desc: t("journey2020"),
              },
              {
                age: 15,
                year: 2021,
                title: "EXPLOIT_MALWARE_INIT",
                desc: t("journey2021"),
              },
              {
                age: 16,
                year: 2022,
                title: "LINUX_AWAKENING",
                desc: t("journey2022"),
              },
              {
                age: 18,
                year: 2025,
                title: "Trying to grind more & more !",
                desc: t("journey2025"),
              },
            ].map((item, index) => (
              <Card
                key={index}
                className="bg-black/80 border-2 border-cyan-400/50 backdrop-blur-sm relative overflow-hidden hover:border-cyan-400/80 transition-all duration-300"
              >
                <div className="absolute inset-0 bg-gradient-to-br from-cyan-400/10 to-purple-400/10"></div>
                <div className="absolute top-0 left-0 w-full h-px bg-gradient-to-r from-cyan-400 to-purple-400 animate-pulse"></div>
                <CardContent className="p-6 relative z-10">
                  <div className="flex items-center mb-4">
                    <div className="bg-gradient-to-r from-cyan-600 to-purple-400 text-white rounded-full w-10 h-10 flex items-center justify-center font-bold mr-4 font-mono shadow-[0_0_20px_rgba(0,255,255,0.3)]">
                      {item.age}
                    </div>
                    <h3 className="text-cyan-400 font-semibold text-lg font-mono">
                      <span className="text-cyan-400">[</span>
                      {item.title}
                      <span className="text-cyan-400">]</span> ({item.year})
                    </h3>
                  </div>
                  <p className="text-gray-300 font-mono text-sm leading-relaxed">{item.desc}</p>
                </CardContent>
              </Card>
            ))}
          </div>

          <div className="text-center mt-8">
            <div className="bg-black/80 border border-cyan-400/50 rounded-lg p-4 backdrop-blur-sm">
              <p className="text-gray-400 italic font-mono text-sm">
                <span className="text-cyan-400">[RÉSUMÉ]</span> "{t("journeySummary")}"
              </p>
            </div>
          </div>
        </div>
      </section>

      {/* Skills Section */}
      <section className="py-20 px-4 bg-black/50 relative z-10">
        <div className="max-w-6xl mx-auto">
          <h2 className="text-4xl font-bold text-center mb-12 text-white font-mono">
            <span className="text-cyan-400">[</span>
            {t("skillsTitle")}
            <span className="text-cyan-400">]</span>
          </h2>

          <div className="grid md:grid-cols-3 gap-8 mb-16">
            {[
              {
                title: t("frontendNinja"),
                icon: Code,
                color: "blue",
                skills: ["HTML5", "CSS3", "JavaScript"],
                desc: t("frontendDesc"),
              },
              {
                title: t("backendSorcier"),
                icon: Server,
                color: "green",
                skills: ["PHP", "Node.js", "Python", "C"],
                desc: t("backendDesc"),
              },
              {
                title: t("linuxAddict"),
                icon: Heart,
                color: "red",
                skills: ["Arch BTW", "Ubuntu", "Debian", "Terminal"],
                desc: t("linuxDesc"),
              },
            ].map((category, index) => (
              <Card
                key={index}
                className="bg-black/80 border-2 border-cyan-400/50 backdrop-blur-sm hover:border-cyan-400/80 transition-all duration-300 relative overflow-hidden group"
              >
                <div className="absolute inset-0 bg-gradient-to-br from-cyan-400/10 to-cyan-600/10 group-hover:from-cyan-400/20 group-hover:to-cyan-600/20 transition-all duration-300"></div>
                <div className="absolute top-0 left-0 w-full h-px bg-gradient-to-r from-cyan-400 to-cyan-600 animate-pulse"></div>
                <CardHeader>
                  <CardTitle className="text-cyan-400 flex items-center font-mono">
                    <category.icon className="w-6 h-6 mr-2 animate-pulse" />
                    <span className="text-cyan-400">[</span>
                    {category.title}
                    <span className="text-cyan-400">]</span>
                  </CardTitle>
                </CardHeader>
                <CardContent className="relative z-10">
                  <div className="space-y-2 mb-4">
                    {category.skills.map((skill, skillIndex) => (
                      <Badge
                        key={skillIndex}
                        variant="secondary"
                        className="bg-cyan-600/20 text-cyan-400 border border-cyan-400/30 font-mono"
                      >
                        {skill}
                      </Badge>
                    ))}
                  </div>
                  <p className="text-gray-400 text-sm font-mono">{category.desc}</p>
                </CardContent>
              </Card>
            ))}
          </div>

          {/* Advanced Skills Grid */}
          <div className="mt-16">
            <h3 className="text-2xl sm:text-3xl font-bold text-center mb-8 text-white font-mono">
              <span className="text-cyan-400">[</span>
              {t("advancedTitle")}
              <span className="text-cyan-400">]</span>
            </h3>
            <div className="grid grid-cols-1 md:grid-cols-2 xl:grid-cols-3 gap-6">
              {[
                {
                  title: t("scrapingMaster"),
                  icon: Zap,
                  color: "cyan",
                  badges: [
                    "Puppeteer Extra Stealth",
                    "Custom Scrapers",
                    "V8 Engine Execution",
                    "Human Simulation",
                    "Web Exploit",
                  ],
                  description:
                    "Puppeteer Stealth pour les cas classiques, mais quand les sites flaggent les librairies connues, je développe mes propres scrapers from scratch. Simulation parfaite du comportement humain, gestion avancée des headers, exécution du moteur V8 pour contourner les protections JS...",
                  warning: true,
                },
                {
                  title: t("aiVocal"),
                  icon: Shield,
                  color: "yellow",
                  badges: ["Whisper + WebRTC", "GPT-4 + ElevenLabs", "Real-time Voice"],
                  description:
                    "Agents vocaux IA avec Whisper pour la reconnaissance, GPT-4 pour l'intelligence et ElevenLabs pour des voix ultra-réalistes. WebRTC pour du temps réel parfait.",
                },
                {
                  title: t("ecommerce"),
                  icon: Terminal,
                  color: "pink",
                  badges: ["Code 100% Custom", "Real-time Data", "Auto-pricing"],
                  description:
                    "E-commerce basé sur du scraping temps réel. Prix, stocks, descriptions automatiquement synchronisés. Tout codé à la main, pas de WordPress qui rame.",
                },
                {
                  title: t("automation"),
                  icon: Server,
                  color: "indigo",
                  badges: ["Google Calendar API", "IA Répondeur", "Auto RDV"],
                  description:
                    "Répondeur IA connecté directement à Google Calendar. L'IA prend les appels, comprend la demande et programme automatiquement les rendez-vous. Zéro intervention humaine.",
                },
                {
                  title: t("apiDev"),
                  icon: Code,
                  color: "orange",
                  badges: ["Express.js", "Node.js", "Rate Limiting"],
                  description:
                    "APIs robustes avec Express.js et Node.js, authentification JWT, rate limiting intelligent. Architecture scalable pour une performance maximale.",
                },
                {
                  title: t("devops"),
                  icon: Heart,
                  color: "red",
                  badges: ["Docker Swarm", "Nginx + SSL", "CI/CD Pipeline"],
                  description:
                    "Infrastructure Linux bulletproof avec monitoring 24/7. Uptime : 99.99% (contrairement à Windows qui redémarre pour installer Paint 3D) 🐧",
                },
              ].map((skill, index) => (
                <Card
                  key={index}
                  className="bg-black/80 border-2 border-cyan-400/50 backdrop-blur-sm hover:border-cyan-400/80 transition-all duration-300 relative overflow-hidden group"
                >
                  <div className="absolute inset-0 bg-gradient-to-br from-cyan-400/10 to-cyan-600/10 group-hover:from-cyan-400/20 group-hover:to-cyan-600/20 transition-all duration-300"></div>
                  <div className="absolute top-0 left-0 w-full h-px bg-gradient-to-r from-cyan-400 to-cyan-600 animate-pulse"></div>

                  <CardHeader>
                    <CardTitle className="text-cyan-400 flex items-center text-lg font-mono animate-pulse">
                      <skill.icon className="w-5 h-5 sm:w-6 sm:h-6 mr-2" />
                      <span className="text-cyan-400">[</span>
                      {skill.title}
                      <span className="text-cyan-400">]</span>
                    </CardTitle>
                  </CardHeader>

                  <CardContent className="relative z-10">
                    <div className="space-y-2 mb-4">
                      {skill.badges.map((badge, badgeIndex) => (
                        <Badge
                          key={badgeIndex}
                          variant="secondary"
                          className="bg-cyan-600/20 text-cyan-400 text-xs border border-cyan-400/30 font-mono"
                        >
                          {badge}
                        </Badge>
                      ))}
                    </div>
                    <p className="text-gray-400 text-sm mb-3 font-mono leading-relaxed">{skill.description}</p>
                    {skill.warning && (
                      <div className="bg-red-900/20 border border-red-500/30 rounded-lg p-3">
                        <p className="text-red-400 text-xs font-semibold mb-1 font-mono">
                          <span className="text-cyan-400">[WARNING]</span> Niveau Expert
                        </p>
                        <p className="text-gray-400 text-xs font-mono">{t("projectWarning")}</p>
                      </div>
                    )}
                  </CardContent>
                </Card>
              ))}
            </div>
          </div>
        </div>
      </section>

      {/* Projects Section */}
      <section className="py-20 px-4 sm:px-6 lg:px-8 relative z-10">
        <div className="max-w-6xl mx-auto">
          <h2 className="text-3xl sm:text-4xl font-bold text-center mb-12 text-white font-mono">
            <span className="text-cyan-400">[</span>
            {t("projectTitle")}
            <span className="text-cyan-400">]</span>
          </h2>

          <div className="max-w-4xl mx-auto">
            <Card className="bg-black/80 border-2 border-purple-400/50 backdrop-blur-sm relative overflow-hidden">
              <div className="absolute inset-0 bg-gradient-to-br from-purple-400/10 to-pink-400/10"></div>
              <div className="absolute top-0 left-0 w-full h-px bg-gradient-to-r from-purple-400 to-pink-400 animate-pulse"></div>
              <CardHeader>
                <CardTitle className="text-purple-400 text-xl sm:text-2xl font-mono">
                  <span className="text-cyan-400">[</span>ScrapeTheMatrix.py<span className="text-cyan-400">]</span>
                </CardTitle>
                <CardDescription className="text-gray-400 text-base sm:text-lg font-mono">
                  <span className="text-cyan-400">&gt;</span> {t("projectSubtitle")}
                </CardDescription>
              </CardHeader>
              <CardContent className="space-y-6 relative z-10">
                <div className="relative">
                  <div className="absolute -inset-2 bg-gradient-to-r from-purple-400/20 to-pink-400/20 rounded-lg blur-xl"></div>
                  <Image
                    src="/images/dev-life.jpeg"
                    alt="Data scraping master at work"
                    width={600}
                    height={300}
                    className="rounded-lg w-full object-cover relative z-10 border border-purple-400/50"
                  />
                </div>

                <div className="space-y-4">
                  <h4 className="text-green-400 font-semibold text-lg font-mono">
                    <span className="text-cyan-400">[</span>
                    {t("projectTargets")}
                    <span className="text-cyan-400">]</span> 🎯
                  </h4>
                  <div className="grid grid-cols-2 sm:grid-cols-3 md:grid-cols-5 gap-3">
                    {["Google", "Amazon", "LinkedIn", "Leboncoin", "Google Maps"].map((target, index) => (
                      <Badge
                        key={index}
                        className="bg-gradient-to-r from-red-600/20 to-orange-600/20 text-red-400 justify-center py-2 border border-red-400/30 font-mono shadow-[0_0_10px_rgba(255,0,0,0.3)]"
                      >
                        {target}
                      </Badge>
                    ))}
                  </div>

                  <div className="bg-black/60 rounded-lg p-4 border border-cyan-400/30">
                    <p className="text-gray-300 text-base leading-relaxed font-mono">
                      <strong className="text-cyan-400">[SYSTÈME_AVANCÉ]</strong> {t("projectAdvanced")}
                    </p>
                  </div>

                  <div className="bg-red-900/20 rounded-lg p-4 border border-red-400/30">
                    <h4 className="text-red-400 font-semibold text-sm mb-2 font-mono">
                      <span className="text-cyan-400">[CLASSIFIED]</span> 🔥 Techniques Avancées (Chut...) :
                    </h4>
                    <ul className="text-gray-400 text-xs space-y-1 font-mono">
                      <li>
                        • <strong>Extra Stealth</strong> : Désactivation complète des signatures webdriver détectables
                      </li>
                      <li>
                        • <strong>Real Canvas</strong> : Génération de vrais canvas pour contourner les empreintes
                      </li>
                      <li>
                        • <strong>Human Movement</strong> : Simulation parfaite des mouvements humains et timing
                      </li>
                      <li>
                        • <strong>Captcha Bypass</strong> : Contournement automatique des systèmes de protection
                      </li>
                      <li>
                        • <strong>Proxy Rotation</strong> : Rotation intelligente et géolocalisée des proxies
                      </li>
                      <li>
                        • <strong>Fake Extensions</strong> : Installation d'extensions pour crédibiliser le navigateur
                      </li>
                      <li>
                        • <strong>Trust Factor Testing</strong> : Tests continus de confiance des techniques de scraping
                      </li>
                    </ul>
                    <p className="text-red-300 text-xs mt-3 italic font-mono">
                      <span className="text-cyan-400">[NOTE]</span> "{t("projectWarning")}"
                    </p>
                  </div>

                  <div className="bg-black/80 rounded-lg p-4 border border-green-400/50">
                    <p className="text-green-400 font-mono text-sm mb-2">
                      <span className="text-cyan-400">$</span> docker-compose up --scale scraper=50
                    </p>
                    <p className="text-gray-400 text-sm font-mono">
                      <span className="text-cyan-400">[RÉSULTAT]</span> {t("projectResult")}
                    </p>
                  </div>

                  <h4 className="text-yellow-400 font-semibold text-lg font-mono">
                    <span className="text-cyan-400">[</span>
                    {t("projectTech")}
                    <span className="text-cyan-400">]</span> ⚡
                  </h4>
                  <div className="flex flex-wrap gap-2">
                    {[
                      "Puppeteer Extra Stealth",
                      "Docker + Kubernetes",
                      "Proxy Rotation",
                      "Anti-Detection",
                      "CAPTCHA Solving",
                      "Data Pipeline",
                    ].map((tech, index) => (
                      <Badge
                        key={index}
                        className="bg-gradient-to-r from-blue-600/20 to-purple-600/20 text-blue-400 border border-blue-400/30 font-mono"
                      >
                        {tech}
                      </Badge>
                    ))}
                  </div>

                  <div className="bg-gradient-to-r from-purple-500/10 to-pink-500/10 rounded-lg p-4 border border-purple-400/30">
                    <p className="text-purple-400 font-semibold mb-2 font-mono">
                      <span className="text-cyan-400">[STATUS]</span> 👑 {t("projectKing")}
                    </p>
                    <ul className="text-gray-300 text-sm space-y-1 font-mono">
                      <li>• Extraction de millions de données par jour sans se faire détecter</li>
                      <li>• Contournement de tous les systèmes anti-bot (Cloudflare, reCAPTCHA, etc.)</li>
                      <li>• Architecture scalable qui peut gérer n'importe quel volume</li>
                      <li>• Données structurées et nettoyées automatiquement</li>
                    </ul>
                  </div>

                  <div className="text-center border-t border-gray-700 pt-4">
                    <p className="text-gray-400 text-sm italic font-mono">
                      <span className="text-cyan-400">[FACT]</span> "{t("projectFact")}" 🐧
                    </p>
                  </div>
                </div>
              </CardContent>
            </Card>
          </div>
        </div>
      </section>

      {/* Contact Section */}
      <section className="py-20 px-4 bg-black/50 relative z-10">
        <div className="max-w-4xl mx-auto text-center">
          <h2 className="text-4xl font-bold mb-8 text-white font-mono">
            <span className="text-cyan-400">[</span>
            {t("contactTitle")}
            <span className="text-cyan-400">]</span>
          </h2>

          <Card className="bg-black/80 border-2 border-green-400/50 backdrop-blur-sm relative overflow-hidden">
            <div className="absolute inset-0 bg-gradient-to-br from-green-400/10 to-cyan-400/10"></div>
            <div className="absolute top-0 left-0 w-full h-px bg-gradient-to-r from-green-400 to-cyan-400 animate-pulse"></div>
            <CardContent className="p-8 relative z-10">
              <div className="mb-6">
                <div className="relative mb-4">
                  <Zap className="w-12 h-12 mx-auto text-yellow-400 animate-pulse drop-shadow-[0_0_20px_rgba(255,255,0,0.5)]" />
                  <div className="absolute -inset-4 bg-yellow-400/20 rounded-full blur-xl animate-pulse"></div>
                </div>
                <p className="text-xl text-gray-300 mb-4 font-mono">
                  <span className="text-cyan-400">[QUERY]</span> {t("contactQuery")}
                </p>
                <p className="text-green-400 font-mono">
                  <span className="text-cyan-400">$</span> sudo apt install freedom-from-microsoft
                </p>
              </div>

              <div className="space-y-4">
                <div className="flex flex-col sm:flex-row gap-4 justify-center">
                  <Button className="bg-gradient-to-r from-green-600 to-cyan-600 hover:from-green-700 hover:to-cyan-700 text-white border border-green-400/50 shadow-[0_0_20px_rgba(0,255,0,0.3)] hover:shadow-[0_0_30px_rgba(0,255,0,0.5)] transition-all duration-300 font-mono">
                    <Terminal className="w-5 h-5 mr-2" />
                    zarrouguidev@gmail.com
                  </Button>
                  <Button
                    variant="outline"
                    className="border-purple-400 text-purple-400 hover:bg-purple-400 hover:text-white bg-transparent shadow-[0_0_20px_rgba(128,0,128,0.3)] hover:shadow-[0_0_30px_rgba(128,0,128,0.5)] transition-all duration-300 font-mono"
                  >
                    <Heart className="w-5 h-5 mr-2" />
                    @zarrouguidev
                  </Button>
                  <Button
                    variant="outline"
                    className="border-blue-400 text-blue-400 hover:bg-blue-400 hover:text-white bg-transparent shadow-[0_0_20px_rgba(0,0,255,0.3)] hover:shadow-[0_0_30px_rgba(0,0,255,0.5)] transition-all duration-300 font-mono"
                  >
                    <Zap className="w-5 h-5 mr-2" />
                    Discord: zarrouguidev
                  </Button>
                </div>

                <div className="space-y-2">
                  <p className="text-gray-400 text-sm font-mono">
                    <span className="text-cyan-400">[STATUS]</span> {t("contactStatus")}
                  </p>
                  <p className="text-gray-500 text-xs font-mono">
                    <span className="text-green-400">[INFO]</span> {t("contactInfo")}
                  </p>
                  <p className="text-cyan-400 text-sm mt-4 font-semibold font-mono border border-cyan-400/30 p-2 rounded">
                    <span className="text-yellow-400">[LEVEL]</span> 🚀 {t("contactLevel")}
                  </p>
                  <p className="text-green-400 text-xs mt-2 font-mono border border-green-400/30 p-2 rounded">
                    <span className="text-cyan-400">[TARGET]</span> 💰 {t("contactTarget")}
                  </p>
                </div>
              </div>
            </CardContent>
          </Card>
        </div>
      </section>

      {/* Footer */}
      <footer className="py-8 px-4 border-t border-cyan-400/30 relative z-10">
        <div className="max-w-6xl mx-auto text-center">
          <p className="text-gray-400 mb-2 font-mono">
            <span className="text-cyan-400">[COPYRIGHT]</span> © {new Date().getFullYear()} Naïm.dev - {t("footerCopyright")}
          </p>
          <p className="text-sm text-gray-500 font-mono">
            <span className="text-green-400">[DISCLAIMER]</span> {t("footerDisclaimer")}
          </p>
        </div>
      </footer>
    </div>
  )
}
