"use client"

import { Badge } from "@/components/ui/badge"
import { Button } from "@/components/ui/button"
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import { Terminal, Code, Server, Shield, Heart, Zap, Github, ExternalLink, ArrowLeft, ArrowRight } from "lucide-react"
import Image from "next/image"
import Link from "next/link"
import { useEffect, useState } from "react"
import { LanguageSelector } from "@/components/LanguageSelector"
import { useTranslation } from "@/hooks/useTranslation"

export default function ProjectsPage() {
  const { language, mounted } = useTranslation()
  const [matrixRain, setMatrixRain] = useState<string[]>([])

  useEffect(() => {
    // Matrix rain effect
    const matrixChars = "01アイウエオカキクケコサシスセソタチツテトナニヌネノハヒフヘホマミムメモヤユヨラリルレロワヲン"
    const rainInterval = setInterval(() => {
      const newRain = Array.from({ length: 20 }, () => matrixChars[Math.floor(Math.random() * matrixChars.length)])
      setMatrixRain(newRain)
    }, 200)

    return () => {
      clearInterval(rainInterval)
    }
  }, [])

  const projects = [
    {
      title: "Login API with JWT",
      description: "API d'authentification sécurisée avec gestion des tokens JWT, système de refresh tokens et protection des routes",
      tech: ["JavaScript", "Node.js"],
      githubUrl: "https://github.com/Chopinetic/login-api-with-jwt",
      liveUrl: null,
      type: "github" as const
    },
    {
      title: "Google Maps Scraper",
      description: "Outil de scraping puissant pour extraire les données des établissements Google Maps avec géolocalisation",
      tech: ["JavaScript", "Node.js"],
      githubUrl: "https://github.com/Chopinetic/google-maps-scraper",
      liveUrl: null,
      type: "github" as const
    },
    {
      title: "ElevenLabs Calling API",
      description: "Intégration API ElevenLabs pour la synthèse vocale et les appels automatisés avec intelligence artificielle",
      tech: ["JavaScript", "Node.js"],
      githubUrl: "https://github.com/Chopinetic/elevenlabs-calling-api",
      liveUrl: null,
      type: "github" as const
    }
  ];

  const isRTL = language === "ar"

  return (
    <div className={`min-h-screen bg-black relative overflow-hidden ${isRTL ? "rtl" : "ltr"}`}>
      <LanguageSelector />

      {/* Matrix Rain Background */}
      <div className="fixed inset-0 pointer-events-none z-0">
        <div className="absolute inset-0 bg-gradient-to-br from-cyan-900/20 via-purple-900/20 to-pink-900/20"></div>
        {matrixRain.map((char, i) => (
          <div
            key={i}
            className="absolute text-green-400/30 font-mono text-sm animate-pulse"
            style={{
              left: `${(i * 5) % 100}%`,
              top: `${Math.random() * 100}%`,
              animationDelay: `${i * 0.1}s`,
            }}
          >
            {char}
          </div>
        ))}
      </div>

      {/* Cyber Grid Overlay */}
      <div className="fixed inset-0 pointer-events-none z-0">
        <div className="absolute inset-0 bg-[linear-gradient(rgba(0,255,255,0.03)_1px,transparent_1px),linear-gradient(90deg,rgba(0,255,255,0.03)_1px,transparent_1px)] bg-[size:50px_50px]"></div>
      </div>

      {/* Main Content */}
      <div className="relative z-10 py-20 px-4">
        <div className="max-w-6xl mx-auto">
          {/* Header */}
          <div className="text-center mb-16">
            <Link href="/">
              <Button
                variant="outline"
                className="mb-8 border-cyan-400 text-cyan-400 hover:bg-cyan-400 hover:text-black bg-transparent shadow-[0_0_20px_rgba(0,255,255,0.3)] hover:shadow-[0_0_30px_rgba(0,255,255,0.5)] transition-all duration-300 font-mono"
              >
                <ArrowLeft className="w-5 h-5 mr-2" />
                [RETOUR_SYSTÈME]
              </Button>
            </Link>

            <div className="mb-8">
              <div className="relative mb-4">
                <Terminal className="w-16 h-16 mx-auto mb-4 text-cyan-400 animate-pulse drop-shadow-[0_0_20px_rgba(0,255,255,0.5)]" />
                <div className="absolute -inset-4 bg-cyan-400/20 rounded-full blur-xl animate-pulse"></div>
              </div>
            </div>

            <h1 className="text-5xl md:text-7xl font-bold mb-6 text-white font-mono">
              <span className="text-cyan-400">[</span>
              <span className="bg-gradient-to-r from-cyan-400 via-purple-400 to-pink-400 bg-clip-text text-transparent animate-pulse">
                MES_PROJETS
              </span>
              <span className="text-cyan-400">]</span>
            </h1>

            <p className="text-xl text-gray-300 mb-8 font-mono max-w-3xl mx-auto">
              <span className="text-cyan-400">[INFO]</span> Portfolio de mes créations techniques - 
              <span className="text-green-400"> GitHub</span> & 
              <span className="text-purple-400"> Projets Clients</span>
            </p>
          </div>

          {/* Projects Grid */}
          <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-8">
            {projects.map((project, index) => (
              <Card key={index} className="bg-black/80 border-2 border-cyan-400/50 backdrop-blur-sm relative overflow-hidden group hover:border-purple-400/70 transition-all duration-300 min-h-[400px] flex flex-col">
                <div className="absolute inset-0 bg-gradient-to-br from-cyan-400/10 to-purple-400/10 group-hover:from-purple-400/10 group-hover:to-pink-400/10 transition-all duration-300"></div>
                <div className="absolute top-0 left-0 w-full h-px bg-gradient-to-r from-cyan-400 to-purple-400 animate-pulse"></div>
                
                <CardHeader className="relative z-10 pb-4">
                  <div className="flex items-center justify-between mb-3">
                    <Badge variant="outline" className={`font-mono text-xs px-3 py-1 ${
                      project.type === 'github' 
                        ? 'border-green-400 text-green-400' 
                        : 'border-purple-400 text-purple-400'
                    }`}>
                      {project.type === 'github' ? '[GITHUB]' : '[CLIENT]'}
                    </Badge>
                    <div className="flex gap-2">
                      {project.type === 'github' ? (
                        <Github className="w-5 h-5 text-green-400 animate-pulse" />
                      ) : (
                        <Server className="w-5 h-5 text-purple-400 animate-pulse" />
                      )}
                    </div>
                  </div>
                  <CardTitle className="text-cyan-400 font-mono text-xl group-hover:text-purple-400 transition-colors duration-300 leading-tight mb-2">
                    {project.title}
                  </CardTitle>
                </CardHeader>

                <CardContent className="relative z-10 flex flex-col flex-1 pt-0 px-6 pb-6">
                  <CardDescription className="text-gray-300 mb-6 font-mono text-sm leading-relaxed min-h-[60px]">
                    {project.description}
                  </CardDescription>

                  <div className="flex flex-wrap gap-2 mb-8">
                    {project.tech.map((tech, techIndex) => (
                      <Badge key={techIndex} variant="secondary" className="text-xs font-mono bg-gray-800/50 text-cyan-300 border border-cyan-400/30 px-2 py-1">
                        {tech}
                      </Badge>
                    ))}
                  </div>

                  <div className="flex gap-3 mt-auto">
                    <Button
                      asChild
                      size="default"
                      className={`flex-1 font-mono text-sm h-10 ${
                        project.type === 'github'
                          ? 'bg-gradient-to-r from-green-600 to-cyan-600 hover:from-green-700 hover:to-cyan-700 border-green-400/50'
                          : 'bg-gradient-to-r from-purple-600 to-pink-600 hover:from-purple-700 hover:to-pink-700 border-purple-400/50'
                      } text-white border shadow-[0_0_20px_rgba(0,255,0,0.3)] hover:shadow-[0_0_30px_rgba(0,255,0,0.5)] transition-all duration-300`}
                    >
                      <a href={project.githubUrl} target="_blank" rel="noopener noreferrer" className="flex items-center justify-center w-full h-full">
                        <Github className="w-4 h-4 mr-2" />
                        [CODE]
                      </a>
                    </Button>
                    
                    {project.liveUrl && (
                      <Button
                        asChild
                        size="default"
                        variant="outline"
                        className="border-cyan-400 text-cyan-400 hover:bg-cyan-400 hover:text-black bg-transparent shadow-[0_0_20px_rgba(0,255,255,0.3)] hover:shadow-[0_0_30px_rgba(0,255,255,0.5)] transition-all duration-300 font-mono text-sm h-10 px-4"
                      >
                        <a href={project.liveUrl} target="_blank" rel="noopener noreferrer" className="flex items-center justify-center w-full h-full">
                          <ExternalLink className="w-4 h-4 mr-2" />
                          [DEMO]
                        </a>
                      </Button>
                    )}
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>

          {/* Cyberpunk Dev Image Section */}
          <div className="mt-16 relative">
            <Card className="bg-black/80 border-2 border-cyan-400/50 backdrop-blur-sm relative overflow-hidden">
              <div className="absolute inset-0 bg-gradient-to-br from-cyan-400/10 to-purple-400/10"></div>
              <div className="absolute top-0 left-0 w-full h-px bg-gradient-to-r from-cyan-400 to-purple-400 animate-pulse"></div>
              
              <CardContent className="relative z-10 p-8 text-center">
                <div className="mb-6">
                  <img 
                    src="/serialexperimentslain1.webp" 
                    alt="Serial Experiments Lain - Cyberpunk Developer" 
                    className="w-full max-w-2xl mx-auto rounded-lg border-2 border-cyan-400/30 shadow-[0_0_30px_rgba(0,255,255,0.3)] hover:shadow-[0_0_50px_rgba(0,255,255,0.5)] transition-all duration-300"
                  />
                </div>
                <h3 className="text-2xl font-mono text-cyan-400 mb-4 animate-pulse">
                  [DÉVELOPPEMENT_EN_COURS]
                </h3>
                <p className="text-gray-300 font-mono mb-6 leading-relaxed">
                  De nombreux projets sont actuellement en développement...<br/>
                  <span className="text-purple-400">// Plus de créations techniques seront bientôt disponibles</span><br/>
                  <span className="text-yellow-400">⚠️ Certains ne peuvent pas être rendus publics tels quels et doivent être modifiés</span>
                </p>
                <Button
                  asChild
                  size="lg"
                  className="bg-gradient-to-r from-purple-600 to-cyan-600 hover:from-purple-700 hover:to-cyan-700 text-white font-mono shadow-[0_0_20px_rgba(128,0,255,0.3)] hover:shadow-[0_0_30px_rgba(128,0,255,0.5)] transition-all duration-300 px-8"
                >
                  <a href="https://github.com/Chopinetic" target="_blank" rel="noopener noreferrer" className="flex items-center">
                    <Github className="w-5 h-5 mr-3" />
                    [EXPLORER_GITHUB]
                  </a>
                </Button>
              </CardContent>
            </Card>
          </div>

          {/* Contact Section */}
          <div className="mt-20 text-center">
            <Card className="bg-black/80 border-2 border-green-400/50 backdrop-blur-sm relative overflow-hidden max-w-2xl mx-auto">
              <div className="absolute inset-0 bg-gradient-to-br from-green-400/10 to-cyan-400/10"></div>
              <div className="absolute top-0 left-0 w-full h-px bg-gradient-to-r from-green-400 to-cyan-400 animate-pulse"></div>
              <CardContent className="p-8 relative z-10">
                <div className="mb-6">
                  <div className="relative mb-4">
                    <Zap className="w-12 h-12 mx-auto text-yellow-400 animate-pulse drop-shadow-[0_0_20px_rgba(255,255,0,0.5)]" />
                    <div className="absolute -inset-4 bg-yellow-400/20 rounded-full blur-xl animate-pulse"></div>
                  </div>
                  <p className="text-xl text-gray-300 mb-4 font-mono">
                    <span className="text-cyan-400">[COLLABORATION]</span> Intéressé par un projet ?
                  </p>
                  <p className="text-green-400 font-mono">
                    <span className="text-cyan-400">$</span> git clone https://github.com/votre-expertise
                  </p>
                </div>

                <Button
                  asChild
                  className="bg-gradient-to-r from-green-600 to-cyan-600 hover:from-green-700 hover:to-cyan-700 text-white border border-green-400/50 shadow-[0_0_20px_rgba(0,255,0,0.3)] hover:shadow-[0_0_30px_rgba(0,255,0,0.5)] transition-all duration-300 font-mono"
                >
                  <Link href="/">
                    <Terminal className="w-5 h-5 mr-2" />
                    [CONTACT_SYSTÈME]
                  </Link>
                </Button>
              </CardContent>
            </Card>
          </div>
        </div>
      </div>

      {/* Footer */}
      <footer className="py-8 px-4 border-t border-cyan-400/30 relative z-10">
        <div className="max-w-6xl mx-auto text-center">
          <p className="text-gray-400 mb-2 font-mono">
            <span className="text-cyan-400">[COPYRIGHT]</span> © {new Date().getFullYear()} Naïm.dev - Tous droits réservés.
          </p>
          <p className="text-sm text-gray-500 font-mono">
            <span className="text-green-400">[DISCLAIMER]</span> Portfolio technique - Projets réalisés avec passion
          </p>
        </div>
      </footer>
    </div>
  )
}
