"use client"

import { useState } from "react"
import { Button } from "@/components/ui/button"
import { Card, CardContent } from "@/components/ui/card"
import { Globe, ChevronDown } from "lucide-react"
import { useTranslation, type Language } from "@/hooks/useTranslation"

const languages = [
  { code: "fr" as Language, name: "Français", flag: "🇫🇷", cyber: "FR_SYSTEM" },
  { code: "en" as Language, name: "English", flag: "🇺🇸", cyber: "EN_SYSTEM" },
  { code: "ar" as Language, name: "العربية", flag: "🇸🇦", cyber: "AR_SYSTEM" },
  { code: "ru" as Language, name: "Русский", flag: "🇷🇺", cyber: "RU_SYSTEM" },
  { code: "es" as Language, name: "Español", flag: "🇪🇸", cyber: "ES_SYSTEM" },
]

export function LanguageSelector() {
  const { language, changeLanguage, mounted } = useTranslation()
  const [isOpen, setIsOpen] = useState(false)

  if (!mounted) {
    return (
      <div className="fixed top-4 right-4 z-50">
        <Button disabled className="bg-black/80 border-2 border-cyan-400/50 text-cyan-400 backdrop-blur-sm font-mono">
          <Globe className="w-4 h-4 mr-2" />
          <span className="text-xs">[LOADING...]</span>
        </Button>
      </div>
    )
  }

  const currentLang = languages.find((lang) => lang.code === language)

  return (
    <div className="fixed top-4 right-4 z-50">
      <div className="relative">
        <Button
          onClick={() => setIsOpen(!isOpen)}
          className="bg-black/80 border-2 border-cyan-400/50 text-cyan-400 hover:bg-cyan-400/20 backdrop-blur-sm font-mono shadow-[0_0_20px_rgba(0,255,255,0.3)] hover:shadow-[0_0_30px_rgba(0,255,255,0.5)] transition-all duration-300"
        >
          <Globe className="w-4 h-4 mr-2 animate-pulse" />
          <span className="text-xs">[{currentLang?.cyber}]</span>
          <span className="ml-2 text-lg">{currentLang?.flag}</span>
          <ChevronDown className={`w-4 h-4 ml-2 transition-transform duration-300 ${isOpen ? "rotate-180" : ""}`} />
        </Button>

        {isOpen && (
          <Card className="absolute top-full right-0 mt-2 bg-black/90 border-2 border-cyan-400/50 backdrop-blur-sm min-w-[200px] overflow-hidden">
            <div className="absolute top-0 left-0 w-full h-px bg-gradient-to-r from-cyan-400 to-purple-400 animate-pulse"></div>
            <CardContent className="p-2">
              {languages.map((lang) => (
                <Button
                  key={lang.code}
                  onClick={() => {
                    changeLanguage(lang.code)
                    setIsOpen(false)
                  }}
                  variant="ghost"
                  className={`w-full justify-start font-mono text-sm hover:bg-cyan-400/20 transition-all duration-300 ${
                    language === lang.code
                      ? "bg-cyan-400/20 text-cyan-400 shadow-[0_0_10px_rgba(0,255,255,0.3)]"
                      : "text-gray-400 hover:text-cyan-400"
                  }`}
                  dir={lang.code === "ar" ? "rtl" : "ltr"}
                >
                  <span className="text-lg mr-2">{lang.flag}</span>
                  <span className="flex-1 text-left">[{lang.cyber}]</span>
                  <span className="text-xs opacity-70">{lang.name}</span>
                </Button>
              ))}
            </CardContent>
          </Card>
        )}
      </div>
    </div>
  )
}
